<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Icon List Widget
 */
class Xs_Icon_List_Widget extends Widget_Base {

	/**
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'xs-icon-list';
	}

	/**
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Beautypress Icon List', 'beautypress' );
	}

	/**
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-bullet-list';
	}

	/**
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'beautypress-elements' ];
	}

	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_icon',
			[
				'label' => __( 'Icon List', 'beautypress' ),
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'text', [
				'label' => __( 'Text', 'beautypress' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => __( 'List Item', 'beautypress' ),
				'default' => __( 'List Item', 'beautypress' ),
			]
		);

		$repeater->add_control(
			'icon', [
				'label' => __( 'Icon', 'beautypress' ),
				'type' => Controls_Manager::ICON,
				'label_block' => true,
				'default' => 'fas fa-check',
			]
		);

		$this->add_control(
			'icon_list',
			[
				'label' => '',
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'text' => __( 'List Item #1', 'beautypress' ),
						'icon' => 'fas fa-check',
					],
					[
						'text' => __( 'List Item #2', 'beautypress' ),
						'icon' => 'fas fa-times',
					],
					[
						'text' => __( 'List Item #3', 'beautypress' ),
						'icon' => 'far fa-dot-circle',
					],
				],
				'title_field' => '<i class="{{ icon }}"></i> {{{ text }}}',
			]
		);

		$this->add_control(
			'view',
			[
				'label' => __( 'View', 'beautypress' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_icon_list',
			[
				'label' => __( 'List', 'beautypress' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);


		$this->add_responsive_control(
			'icon_align',
			[
				'label' => __( 'Alignment', 'beautypress' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Start', 'beautypress' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'beautypress' ),
						'icon' => 'eicon-h-align-center',
					],
					'right' => [
						'title' => __( 'End', 'beautypress' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'prefix_class' => 'elementor-align-',
				'classes' => 'elementor-control-start-end',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_icon_style',
			[
				'label' => __( 'Icon', 'beautypress' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'icon_color',
			[
				'label' => __( 'Color', 'beautypress' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .elementor-icon-list-icon i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'icon_border_color',
			[
				'label' => __( 'Border Color', 'beautypress' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .beautypress-simple-iocn-list.beautypress-version-2 ul li i' => 'border: 2px solid {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_text_style',
			[
				'label' => __( 'Text', 'beautypress' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);


		$this->add_control(
			'text_color',
			[
				'label' => __( 'Text Color', 'beautypress' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .elementor-icon-list-text' => 'color: {{VALUE}};',
				],
			]
		);


		$this->end_controls_section();
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings();
		?>
		<div class="beautypress-simple-iocn-list beautypress-version-2">
			<ul class="elementor-icon-list-items">
				<?php foreach ( $settings['icon_list'] as $index => $item ) :
					$repeater_setting_key = $this->get_repeater_setting_key( 'text', 'icon_list', $index );

					$this->add_render_attribute( $repeater_setting_key, 'class', 'elementor-icon-list-text' );

					$this->add_inline_editing_attributes( $repeater_setting_key );
					?>
					<li class="elementor-icon-list-item">
						<?php
						if ( $item['icon'] ) :
						?>
							<span class="elementor-icon-list-icon">
								<i class="<?php echo esc_attr( $item['icon'] ); ?>"></i>
							</span>
						<?php endif; ?>
						<span <?php echo beautypress_return($this->get_render_attribute_string( $repeater_setting_key )); ?>><?php echo esc_html( $item['text'] ); ?></span>
					</li>
					<?php
				endforeach;
				?>
			</ul>
		</div>
		<?php
	}

	protected function content_template() {}
}
